# frozen_string_literal: true

require "roby/droby/logfile"
require "roby/droby/logfile/index"

module Roby
    module DRoby
        module Logfile
            # A class that reads log files generated by {Writer}
            class Reader
                # The current log format version
                FORMAT_VERSION = 5

                attr_reader :event_io

                def initialize(event_io, index_path: nil)
                    @event_io = event_io
                    @index_path = index_path ||
                                  event_io.path.gsub(/\.log$/, "") + ".idx"
                    event_io.rewind
                    options_hash = read_header
                    self.class.process_options_hash(options_hash)
                end

                def read_header
                    Logfile.read_prologue(event_io)
                    if chunk = Logfile.read_one_chunk(event_io)
                        ::Marshal.load(chunk)
                    else
                        raise InvalidFileError,
                              "expected the prologue to be followed by one chunk, "\
                              "but got nothing"
                    end
                end

                def dup
                    Reader.new(event_io.dup)
                end

                def tell
                    event_io.tell
                end

                def close
                    event_io.close
                end

                def eof?
                    event_io.eof?
                end

                def closed?
                    event_io.closed?
                end

                def seek(pos)
                    event_io.seek(pos)
                end

                def load_one_cycle
                    return unless (chunk = Logfile.read_one_chunk(event_io))

                    begin ::Marshal.load_with_missing_constants(chunk)
                    rescue ArgumentError => e
                        if e.message == "marshal data too short"
                            raise TruncatedFileError, "marshal data invalid"
                        end

                        raise
                    end
                rescue Exception => e
                    raise e, "#{e.message}, running roby-log repair might repair the file", e.backtrace
                end

                def self.process_options_hash(options_hash)
                    options_hash[:plugins]&.each do |plugin_name|
                        begin
                            Roby.app.using plugin_name
                        rescue ArgumentError => e
                            Roby.warn "the log file mentions the #{plugin_name} plugin, "\
                                      "but it is not available on this system. "\
                                      "Some information might not be displayed"
                        end
                    end
                end

                # The standard Roby index path, inferred from the log file's own
                # path
                #
                # @return [String]
                attr_reader :index_path

                def rebuild_index(path = index_path)
                    Logfile.warn "rebuilding index file for #{event_io.path}"
                    File.open(path, "w") do |index_io|
                        File.open(event_io.path) do |file_io|
                            Index.rebuild(file_io, index_io)
                        end
                    end
                    @index = nil
                end

                # Returns the index object for this event log
                def index(path = index_path, rebuild: true)
                    if @index
                        return @index
                    elsif !File.file?(path)
                        unless rebuild
                            raise IndexMissing, "there's no file #{path}"
                        end

                        rebuild_index(path)
                    end

                    index =
                        begin Index.read(path)
                        rescue Exception => e
                            raise e, "while reading index file #{path}: #{e.message}", e.backtrace
                        end
                    if index.valid_for?(event_io.path)
                        @index = index
                    elsif !rebuild
                        raise IndexInvalid, "#{path} is not a valid index for #{self}"
                    else
                        rebuild_index(path)
                        @index = Index.read(path)
                    end
                end

                def self.open(path, index_path: nil)
                    io = new(File.open(path), index_path: index_path)
                    if block_given?
                        begin
                            yield(io)
                        ensure
                            io.close unless io.closed?
                        end
                    else
                        io
                    end
                end
            end
        end
    end
end
