#include <rtt/transports/corba/TaskContextProxy.hpp>
#include "TaskItem.hpp"
#include <rtt/typelib/TypelibMarshallerBase.hpp>
#include <lib_config/TypelibConfiguration.hpp>
#include <base-logging/Logging.hpp>

TaskItem::TaskItem(RTT::corba::TaskContextProxy* _task)
    : task(_task),
      nameItem(ItemType::TASK),
      statusItem(ItemType::TASK),
      refreshPorts(true),
      stateChanged(false)
{
    inputPorts.setText("InputPorts");
    outputPorts.setText("OutputPorts");
    properties.setText("Properties");
    nameItem.appendRow(&inputPorts);
    nameItem.appendRow(&outputPorts);
    nameItem.appendRow(&properties);
    nameItem.setData(this);
    statusItem.setData(this);
    statusItem.setText("");
}

TaskItem::~TaskItem()
{
}

void TaskItem::reset()
{
    for (auto port: ports)
    {
        OutputPortItem *outPort = dynamic_cast<OutputPortItem*>(port.second);
        if (outPort)
        {
            outPort->reset();
        }
    }
    
    task = nullptr;
}

bool TaskItem::hasVisualizers()
{
    for (auto &port: ports)
    {
        if (port.second->hasVisualizers())
        {
            return true;
        }
    }
    
    return false;
}

bool TaskItem::update()
{   
    if (!task)
    {
        return false;
    }
    
    if (!task->server() || task->server()->_is_nil())
    {
        LOG_WARN_S << "TaskItem::update(): disconnect of task " << task->getName();
        reset();
        return false;
    }
    
    bool needsUpdate = false;
    if (nameItem.text().isEmpty())
    {
        try
        {
            nameItem.setText(task->getName().c_str());
            needsUpdate = true;
        }
        catch (...)
        {
            return false;
        }
    }
    
    // check for task state update
    stateChanged = updateState();
    needsUpdate |= stateChanged;
    
    // check for property update
    if (propertyMap.empty() || stateChanged)
    {
        needsUpdate |= updateProperties();
    }

    // check for port update
    bool hasVis = hasVisualizers();
    if (!hasVis && !nameItem.isExpanded())
    {
        return needsUpdate;
    }
    
    needsUpdate |= updatePorts(hasVis);

    return needsUpdate;
}

bool TaskItem::updatePorts(bool hasVisualizers)
{
    bool needsUpdate = false;
    bool refreshedOutputPorts = false;
    
    if (ports.empty() || hasVisualizers || outputPorts.isExpanded())
    {
        const RTT::DataFlowInterface *dfi = task->ports();

        for (RTT::base::PortInterface *pi : dfi->getPorts())
        {
            const std::string portName(pi->getName());
            RTT::base::OutputPortInterface *outIf = dynamic_cast<RTT::base::OutputPortInterface *>(pi);
            auto it = ports.find(portName);
            PortItem *item = nullptr;
            if (it == ports.end())
            {
                if (outIf)
                {
                    item = new OutputPortItem(outIf);
                    outputPorts.appendRow(item->getRow());
                    item->getNameItem()->appendRow({new QStandardItem(QString("loading..")), new QStandardItem()});
                }
                else
                {
                    item = new PortItem(pi->getName());
                    inputPorts.appendRow(item->getRow());
                }
                
                ports.insert(std::make_pair(portName, item));
                needsUpdate = true;
            }
            else
            {   
                item = it->second;
            }

            if ((item->hasVisualizers() || outputPorts.isExpanded()) && outIf)
            {
                OutputPortItem *outPortItem = static_cast<OutputPortItem *>(item);
                
                if (refreshPorts)
                {
                    outPortItem->updateOutputPortInterface(outIf);
                    refreshedOutputPorts = true;
                }
                
                needsUpdate |= outPortItem->updataValue();
            }
        }
    }
    
    if (refreshedOutputPorts)
    {
        refreshPorts = false;
    }
    
    return needsUpdate;
}

bool TaskItem::updateProperties()
{   
    bool needsUpdate = false;

    RTT::PropertyBag *taskProperties = task->properties();
    orogen_transports::TypelibMarshallerBase *transport;
    orogen_transports::TypelibMarshallerBase::Handle *transportHandle;
    RTT::base::DataSourceBase::shared_ptr sample;
    
    for (std::size_t i=0; i<taskProperties->size(); i++)
    {
        RTT::base::PropertyBase *property = taskProperties->getItem(i);
        
        RTT::types::TypeInfo const *type = property->getTypeInfo();
        transport = dynamic_cast<orogen_transports::TypelibMarshallerBase *>(type->getProtocol(orogen_transports::TYPELIB_MARSHALLER_ID));
        if (! transport)
        {
            LOG_ERROR_S << "cannot report ports of type " << type->getTypeName() << " as no typekit generated by orogen defines it";
            continue;
        }
            
        const Typelib::Type *typelibType = transport->getRegistry().get(transport->getMarshallingType());
        
        transportHandle = transport->createSample();
        sample = transport->getDataSource(transportHandle);
        transport->readDataSource(*(property->getDataSource()), transportHandle);
        
        Typelib::Value val(transport->getTypelibSample(transportHandle), *(typelibType));
        
        std::shared_ptr<ItemBase> item = nullptr;
        
        auto it = propertyMap.find(property->getName());
        if (it == propertyMap.end())
        {
            item = getItem(val);
            
            propertyMap[property->getName()] = item;
            item->setName(property->getName().c_str());
            
            properties.appendRow(item->getRow());
            needsUpdate = true;
        }
        else
        {
            item = propertyMap[property->getName()];
            needsUpdate |= item->update(val, properties.isExpanded(), true);
        }
    }
    
    return needsUpdate;
}

bool TaskItem::updateState()
{
    std::string stateString = "";
    RTT::base::TaskCore::TaskState state = task->getTaskState();
    switch(state)
    {
        case RTT::base::TaskCore::Exception:
            stateString = "Exception";
            break;
        case RTT::base::TaskCore::FatalError:
            stateString = "FatalError";
            break;
        case RTT::base::TaskCore::Init:
            stateString = "Init";
            break;
        case RTT::base::TaskCore::PreOperational:
            stateString = "PreOperational";
            break;
        case RTT::base::TaskCore::Running:
            stateString = "Running";
            break;
        case RTT::base::TaskCore::RunTimeError:
            stateString = "RunTimeError";
            break;
        case RTT::base::TaskCore::Stopped:
            stateString = "Stopped";
            break;
    }
    
    if (statusItem.text().toStdString() != stateString)
    {
        statusItem.setText(stateString.c_str());
        return true;
    }

    return false;
}

QList< QStandardItem* > TaskItem::getRow()
{
    return {&nameItem, &statusItem};
}

QModelIndex TaskItem::updateLeft()
{
    return nameItem.index();
}

QModelIndex TaskItem::updateRight()
{
    return statusItem.index();
}

RTT::corba::TaskContextProxy* TaskItem::getTaskContext()
{
    return task;
}
