#include <iostream>
#include <algorithm>  // for std::for_each
#include <stdio.h> //for printf
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <imu_stim300/Stim300RevD.hpp>
#include <imu_stim300/Stim300RevB.hpp>


using namespace std;

/**
* SIGNAL      ID   DEFAULT  DESCRIPTION
* ===============================================
*	SIGHUP 		1    Termin.  Hang up on controlling terminal
*  SIGINT			2    Termin.  Interrupt. Generated when we enter CNRTL-C and it is delivered
*								to all processes/threads associated to the current terminal. If generated
*								with kill, it is delivered to only one process/thread.
*  SIGQUIT      3    Core     Generated when at terminal we enter CNRTL-\
*  SIGILL         4    Core     Generated when we executed an illegal instruction
*  SIGTRAP     5    Core     Trace trap (not reset when caught)
*  SIGABRT     6    Core     Generated by the abort function
*  SIGFPE       8    Core     Floating Point error
*  SIGKILL      9    Termin.  Termination (can't catch, block, ignore)
*  SIGBUS     10    Core     Generated in case of hardware fault
*  SIGSEGV   11    Core     Generated in case of illegal address
*  SIGSYS      12    Core     Generated when we use a bad argument in a system service call
*  SIGPIPE     13    Termin.  Generated when writing to a pipe or a socket while no process is reading at other end
*  SIGALRM   14    Termin.  Generated by clock when alarm expires
*  SIGTERM   15    Termin.  Software termination signal
*  SIGURG     16    Ignore   Urgent condition on IO channel
*  SIGCHLD   20    Ignore   A child process has terminated or stopped
*  SIGTTIN     21    Stop     Generated when a backgorund process reads from terminal
*  SIGTTOUT  22    Stop     Generated when a background process writes to terminal
*  SIGXCPU    24    Discard  CPU time has expired
*  SIGUSR1    30    Termin.  User defiled signal 1
*  SIGUSR2    31    Termin.  User defined signal 2
*
* @author Javier Hidalgo Carrio .
*/

/**
* @brief POINTER TO HANDLING FUNCTION
*
* Defines for the signaling library. Definitiod of data type for pointer to handling function
*/
typedef void (* handling_t) (int sig, void *data);

/**
* @brief GLOBAL ARRAY.
*
*  Two arrays:
*  		handling_t htab: one for pointer of function (the handling function)
* 		data: array of handling function parameters. 
*/

static handling_t htab [20] ;
static void *data[20];

/**
* @brief Function that set a generic handler for a signal.
* 
* @author Javier Hidalgo Carrio.
* @param[in] sig integer for the number of signal to be handled
*
* @return Void
*
*/
static void generic_handling (int sig)
{

   htab[sig](sig,data[sig]);

}

/**
* @brief Function that set a defined handler for a signal.
* 
* @author Javier Hidalgo Carrio.
* @param[in] sig integer for the number of signal to be handled
* @param[in] pfunc pointer of the handling function
* @param[in] values pointer of void, the parameter of the handling function.
*
* @return OK is everything all right. ERROR on other cases
*
*/
bool signal_set (int sig,handling_t pfunc,void *values)
{

	htab[sig] = pfunc;
	data[sig] = values;
	if ((signal (sig,generic_handling))==SIG_ERR)
	 return false;
	return true;
}


/**
* @brief The Function catchs the signal introduced in the parameter of this function
* 
* The function associates the signals with the handling function to be handled when
* the process would receives such signal from the operating system.
* The handling function should have two parameters when defined in the main program.
* one parameter (integer) for the signal number, another for the pointer to void (values)
* Normally the last one is a pointer to a structure.
*
* @author Javier Hidalgo Carrio.
* @param[in] sig_1 integer for the number of signal 1 to be handled
* @param[in] sig_2 integer for the number of signal 2 to be handled
* @param[in] sig_3 integer for the number of signal 3 to be handled
* @param[in] sig_4 integer for the number of signal 4 to be handled
* @param[in] pfunc pointer to the handling function
* @param[in] values pointer to void, the parameter of the handling function.
*
* @return OK is everything all right. ERROR in other cases
*
*/
bool signal_catcher (int sig_1,int sig_2,int sig_3, int sig_4, handling_t pfunc, void *values)
{
	bool status=true;

	status=signal_set (sig_1,pfunc,values);

	if (status)

	status=signal_set (sig_2,pfunc,values);

	if (status)

	status=signal_set (sig_3,pfunc,values);

	if (status)

	status=signal_set (sig_4,pfunc,values);
	
	return status;

}

/**
 * @brief Function for handling system signals.
 * 
 * This function is the handler when the catched
 * system signal happen.
 *
 * @author Javier Hidalgo Carrio.
 *
 * @param[in] sig integer of the catched signal
 * @param[in] values pointer of void (pointer of the IMU structure after casting in the function)
 *
 * @return void
 *
 */
void signal_terminator (int sig, void *values)
{
	bool status = true;
	imu_stim300::Stim300Base * pdriver;

	pdriver = (imu_stim300::Stim300Base *) values;

	if (sig == SIGINT)
		std::cout<< "\nSIGINT: Terminal interrupt\n";
	else if (sig == SIGTERM)
		std::cout<< "\nSIGTERM: Termination\n";
	else if (sig == SIGHUP)
		std::cout<< "\nSIGHUP: Hangup\n";
	else std::cout<<"\nSIGSEGV: Segmentation\n";


	
        pdriver->close();
        std::cout<<"serial port closed correctly.\n";
	
  	exit (status);
	
}

int main(int argc, char** argv)
{
    imu_stim300::Stim300RevD myDriverRevD;
    imu_stim300::Stim300RevB myDriverRevB;
    handling_t pfunc;
	
    if (argc<2)
    {
	printf( "Usage: imu_stim300_bin <device>\n");
	return 0;
    }

        /* Function signal handling */
    pfunc = signal_terminator;

    /* Catching system signals */
    signal_catcher (SIGHUP,SIGINT,SIGTERM, SIGSEGV, pfunc, &myDriverRevD);

//     ******************************************
    // This is "123456789" in ASCII
//     unsigned char const  data[] = { 0x97, 0x00, 0x01, 0x37, 0x00, 0x01, 0xc7, 0xff, 0xfe, 0x74,
//      0x00, 0x00, 0x06, 0x32, 0x00, 0x11, 0xad, 0x07, 0xf4, 0x45, 0x00, 0xff, 0xec, 0xbc, 0x00, 0x37, 0x93, 0x40, 0x06, 0x2a, 0x00,
//      0x20, 0x4f, 0x20, 0x69, 0x20, 0x55, 0x00, 0xd9, 0x03, 0xf8};
//     std::size_t const    data_len = sizeof( data ) / sizeof( data[0] );


//     for (int i=0; i<(int)sizeof(data);i++)
//     {
// 	printf("%X \n", data[i]);
//     }
//     std::cout<<"Size of data: "<< sizeof(data)<<"\n";
//     std::cout<<"Size of data: "<< sizeof(data)/sizeof(data[0])<<"\n";

//     // The expected CRC for the given data
//     boost::uint32_t const  expected = 0x21c1e045;

//     // Simulate CRC-CCITT
//     boost::crc_basic<32>  crc_32(0x04C11DB7, 0xFFFFFFFF, 0x00, false, false);
//     crc_32.process_bytes(data, data_len);
//     printf("Checksum: %X \n", crc_32.checksum());
//     printf("Expected: %X \n", expected);
//
//     // Simulate CRC-CCITT
//     crc_32 =  boost::crc_basic<32> (0x04C11DB7, 0xFFFFFFFF, 0x00, true, false);
//     crc_32.reset();
//     crc_32.process_bytes(data, data_len);
//     printf("Checksum: %X \n", crc_32.checksum());
//     printf("Expected: %X \n", expected);
//
//     // Simulate CRC-CCITT
//     crc_32 = boost::crc_basic<32> (0x04C11DB7, 0xFFFFFFFF, 0x00, false, true);
//     crc_32.reset();
//     crc_32.process_bytes(data, data_len);
//     printf("Checksum: %X \n", crc_32.checksum());
//     printf("Expected: %X \n", expected);
//
//
//
//
//     // Simulate CRC-CCITT
//     crc_32 = boost::crc_basic<32> (0x04C11DB7, 0xFFFFFFFF, 0x00, true, true);
//     crc_32.reset();
//     crc_32.process_bytes(data, data_len);
//     printf("Checksum: %X \n", crc_32.checksum());
//     printf("Expected: %X \n", expected);
//
//     std::cout << "All tests passed." << std::endl;
//
//      // This is "123456789" in ASCII
//     unsigned char const  data2[] = { 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
//      0x38, 0x39 };
//     std::size_t const    data2_len = sizeof( data2 ) / sizeof( data2[0] );
//
//     // The expected CRC for the given data
//     boost::uint16_t const  expected2 = 0x29B1;
//
//     // Simulate CRC-CCITT
//     boost::crc_basic<16>  crc_ccitt1( 0x1021, 0xFFFF, 0, false, false );
//     crc_ccitt1.process_bytes( data2, data2_len );
//      printf("Checksum: %X \n", crc_ccitt1.checksum());
//     assert( crc_ccitt1.checksum() == expected2 );
//
//     // Repeat with the optimal version (assuming a 16-bit type exists)
//     boost::crc_optimal<16, 0x1021, 0xFFFF, 0, false, false>  crc_ccitt2;
//     crc_ccitt2 = std::for_each( data2, data2 + data2_len, crc_ccitt2 );
//     assert( crc_ccitt2() == expected2 );
//
//
//     std::cout << "All tests passed." << std::endl;
//     return 0;


    myDriverRevD.welcome();

    myDriverRevD.setBaudrate(iodrivers_base::Driver::SERIAL_921600);


    if (!myDriverRevD.open(argv[1]))
    {
	cerr << "cannot open device: " << argv[1] << endl;
	perror("errno is");
	return 1;
    }

    myDriverRevD.printInfo();

    //int i = 0;

//    while (true)
//    {
        usleep(9800);
	myDriverRevD.processPacket();
	myDriverRevD.printInfo();

	
// 	usleep (2000);
// 	if (i == 800)
// 	{
// 	    std::cout<<"RESET IN NORMAL MODE\n";
// 	    myDriverRevD.fullReset();
// 	}
//	
// 	if (i == 1600)
// 	{
// 	    std::cout<<"ENTER IN SERVICE MODE\n";
// 	    myDriverRevD.enterServiceMode();
//
// 	}
//	
// 	if (i == 1604)
// 	{
// 	    std::cout<<"EXIT SERVICE MODE\n";
// 	    myDriverRevD.fullReset();
// 	}
//	
// 	if (i == 3200)
// 	{
// 	    std::cout<<"ACC TO INCREMENTAL VELOCITY\n";
// 	    myDriverRevD.setAcctoIncrementalVelocity();
// 	}
//	
//	std::cout<<"I: "<<i<<"\n";
//	i++;
	
//    }

    myDriverRevD.close();
	
    return 0;
}
