#include <OpenANN/io/Logger.h>
#include <OpenANN/util/AssertionMacros.h>
#include <iomanip>
#include <ctime>

namespace OpenANN
{

const char* LevelToString[] =
{
  "DISABLED",
  "ERROR",
  "INFO",
  "DEBUG"
};

bool Logger::deactivate = false;

Logger::Logger(Target target, std::string name)
  : target(target), name(name)
{
  if(deactivate)
  {
    this->target = NONE;
  }
  else if(target == FILE)
  {
    time_t now;
    std::time(&now);
    struct tm* current = std::localtime(&now);
    char current_time[32];
    std::strftime(current_time, sizeof(current_time), "%F %X", current);
    file.open((name + "-" + std::string(current_time) + ".log").c_str());
    OPENANN_CHECK(file.is_open());
  }
  else if(target == APPEND_FILE)
  {
    file.open((name + ".log").c_str(), std::fstream::app);
    OPENANN_CHECK(file.is_open());
  }
}

Logger::~Logger()
{
  if(file.is_open())
    file.close();
}

bool Logger::isActive()
{
  return target != NONE && !deactivate;
}

Logger& operator<<(Logger& logger, const FloatingPointFormatter& t)
{
  switch(logger.target)
  {
  case Logger::CONSOLE:
    std::cout << std::fixed << std::setprecision(t.precision) << t.value
              << std::resetiosflags(std::ios_base::fixed) << std::flush;
    break;
  case Logger::APPEND_FILE:
  case Logger::FILE:
    logger.file << std::fixed << std::setprecision(t.precision) << t.value << std::flush;
    break;
  default: // do not log
    break;
  }
  return logger;
}

std::ostream* Log::stream = &std::cout;

Log::Log()
  : level(DEBUG)
{
}

Log::~Log()
{
  getStream() << message.str() << std::endl;
}

std::ostream& Log::get(LogLevel level, const char* name_space)
{
  OPENANN_CHECK(level != DISABLED);

  time_t now;
  std::time(&now);
  struct tm* current = std::localtime(&now);
  char current_time[32];

  this->level = level;

  std::strftime(current_time, sizeof(current_time), "%F %X", current);

  message << std::setw(6) << LevelToString[level] << "  "
          << current_time << "  ";

  if(name_space != NULL)
    message << name_space << ": ";

  return message;
}

void Log::setStream(std::ostream& stream)
{
  Log::stream = &stream;
}

std::ostream& Log::getStream()
{
  return *stream;
}

Log::LogLevel& Log::getLevel()
{
  static Log::LogLevel gLevel = DEBUG;
  return gLevel;
}

void Log::setDisabled()
{
  Log::getLevel() = Log::DISABLED;
}

void Log::setError()
{
  Log::getLevel() = Log::ERROR;
}

void Log::setInfo()
{
  Log::getLevel() = Log::INFO;
}

void Log::setDebug()
{
  Log::getLevel() = Log::DEBUG;
}

std::ostream& operator<<(std::ostream& os, const FloatingPointFormatter& t)
{
  os << std::fixed << std::setprecision(t.precision) << t.value
     << std::resetiosflags(std::ios_base::fixed);
  return os;
}

}
