#include "PluginManager.hpp"
#include <tinyxml.h>
#include <boost/filesystem.hpp>
#include <boost/algorithm/string.hpp>
#include <glog/logging.h>

using namespace plugin_manager;

static const std::string plugin_files_path = "/plugin_manager/";
static const std::string plugin_file_extension = ".xml";

PluginManager::PluginManager(const std::vector< std::string >& plugin_xml_paths,
                             bool load_environment_paths, bool auto_load_xml_files)
{
    std::copy(plugin_xml_paths.begin(), plugin_xml_paths.end(), std::back_inserter(this->plugin_xml_paths));
    if(load_environment_paths)
    {
        std::vector<std::string> env_xml_paths = getPluginXmlPathsFromEnv();
        std::copy(env_xml_paths.begin(), env_xml_paths.end(), std::back_inserter(this->plugin_xml_paths));
    }

    if(auto_load_xml_files)
        reloadXMLPluginFiles();
}

PluginManager::~PluginManager()
{
    clear();
}

std::vector< std::string > PluginManager::getPluginXmlPaths() const
{
    return plugin_xml_paths;
}

std::vector< std::string > PluginManager::getAvailableClasses() const
{
    std::vector<std::string> classes;
    classes.reserve(classes_available.size());
    for(const std::pair<std::string, PluginInfoPtr>& plugin_info : classes_available)
    {
        classes.push_back(plugin_info.second->full_class_name);
    }
    return classes;
}

std::vector< std::string > PluginManager::getAvailableClasses(const std::string& base_class) const
{
    std::vector<std::string> classes;
    classes.reserve(base_classes_available.count(base_class));
    std::pair<std::multimap<std::string, PluginInfoPtr>::const_iterator, std::multimap<std::string, PluginInfoPtr>::const_iterator> range;
    range = base_classes_available.equal_range(base_class);
    for(std::multimap<std::string, PluginInfoPtr>::const_iterator it = range.first; it != range.second; it++)
    {
        classes.push_back(it->second->full_class_name);
    }
    return classes;
}

bool PluginManager::isClassInfoAvailable(const std::string& class_name) const
{
    std::string full_class_name;
    if(getFullClassName(class_name, full_class_name))
        return true;
    return false;
}

bool PluginManager::getBaseClass(const std::string& class_name, std::string& base_class) const
{
    std::string full_class_name;
    if(!getFullClassName(class_name, full_class_name))
        return false;

    std::map<std::string, PluginInfoPtr>::const_iterator plugin_info = classes_available.find(full_class_name);
    if(plugin_info != classes_available.end())
    {
        base_class = plugin_info->second->base_class_name;
        return true;
    }
    return false;
}

bool PluginManager::getAssociatedClasses(const std::string& class_name, std::vector<std::string>& associated_classes) const
{
    std::string full_class_name;
    if(!getFullClassName(class_name, full_class_name))
        return false;

    std::map<std::string, PluginInfoPtr>::const_iterator plugin_info = classes_available.find(full_class_name);
    if(plugin_info != classes_available.end())
    {
        if(plugin_info->second->associated_classes.empty())
            return false;
        else
        {
            associated_classes = plugin_info->second->associated_classes;
            return true;
        }
    }
    return false;
}

bool PluginManager::getClassDescription(const std::string& class_name, std::string& class_description) const
{
    std::string full_class_name;
    if(!getFullClassName(class_name, full_class_name))
        return false;

    std::map<std::string, PluginInfoPtr>::const_iterator plugin_info = classes_available.find(full_class_name);
    if(plugin_info != classes_available.end())
    {
        class_description = plugin_info->second->description;
        return true;
    }
    return false;
}

bool PluginManager::getSingletonFlag(const std::string& class_name, bool& is_singleton) const
{
    std::string full_class_name;
    if(!getFullClassName(class_name, full_class_name))
        return false;

    std::map<std::string, PluginInfoPtr>::const_iterator plugin_info = classes_available.find(full_class_name);
    if(plugin_info != classes_available.end())
    {
        is_singleton = plugin_info->second->singleton;
        return true;
    }
    return false;
}

bool PluginManager::getClassLibraryPath(const std::string& class_name, std::string& library_path) const
{
    std::string full_class_name;
    if(!getFullClassName(class_name, full_class_name))
        return false;

    std::map<std::string, PluginInfoPtr>::const_iterator plugin_info = classes_available.find(full_class_name);
    if(plugin_info != classes_available.end())
    {
        library_path = plugin_info->second->library_path;
        return true;
    }
    return false;
}

bool PluginManager::getAssociatedClassOfType(const std::string& embedded_type, const std::string& base_class_name, std::string& associated_class) const
{
    std::vector<std::string> available_classes = getAvailableClasses(base_class_name);
    for(const std::string& class_name : available_classes)
    {
        std::vector<std::string> associated_classes;
        if(getAssociatedClasses(class_name, associated_classes) && !associated_classes.empty())
        {
            std::vector<std::string>::const_iterator it = std::find(associated_classes.begin(), associated_classes.end(), embedded_type);
            if(it != associated_classes.end())
            {
                associated_class = class_name;
                return true;
            }
        }
    }
    return false;
}

std::set< std::string > PluginManager::getRegisteredLibraries() const
{
    std::set< std::string > registered_libraries;
    for(const std::pair<std::string, PluginInfoPtr>& plugin_info : classes_available)
    {
        registered_libraries.insert(plugin_info.second->library_path);
    }
    return registered_libraries;
}

bool PluginManager::removeClassInfo(const std::string& class_name)
{
    std::string full_class_name;
    if(!getFullClassName(class_name, full_class_name))
        return false;

    std::map<std::string, PluginInfoPtr>::iterator plugin_info = classes_available.find(full_class_name);
    if(plugin_info != classes_available.end())
    {
        std::pair<std::multimap<std::string, PluginInfoPtr>::iterator, std::multimap<std::string, PluginInfoPtr>::iterator> range;
        range = base_classes_available.equal_range(plugin_info->second->base_class_name);
        for(std::multimap<std::string, PluginInfoPtr>::iterator it = range.first; it != range.second; it++)
        {
            if(it->second == plugin_info->second)
                base_classes_available.erase(it);
        }
        range = classes_no_ns_available.equal_range(plugin_info->second->class_name);
        for(std::multimap<std::string, PluginInfoPtr>::iterator it = range.first; it != range.second; it++)
        {
            if(it->second == plugin_info->second)
                classes_no_ns_available.erase(it);
        }
        classes_available.erase(plugin_info);
        return true;
    }
    return false;
}

void PluginManager::clear()
{
    classes_available.clear();
    base_classes_available.clear();
    classes_no_ns_available.clear();
}

void PluginManager::overridePluginXmlPaths(const std::vector< std::string >& plugin_xml_paths)
{
    this->plugin_xml_paths = plugin_xml_paths;
}

void PluginManager::reloadXMLPluginFiles()
{
    std::set<std::string> plugin_xml_files;
    for(const std::string &folder : plugin_xml_paths)
    {
        determineAvailableXMLPluginFiles(folder, plugin_xml_files);
    }
    for(const std::string &file : plugin_xml_files)
    {
        std::vector< PluginManager::PluginInfoPtr > classes;
        if(processSingleXMLPluginFile(file, classes))
            insertPluginInfos(classes);
    }
}

std::vector< std::string > PluginManager::getPluginXmlPathsFromEnv() const
{
    const char* install_path = std::getenv("LD_LIBRARY_PATH");
    std::vector<std::string> paths;
    if(install_path != NULL)
    {
        const std::string path(install_path);
        //":" is the separator in LD_LIBRARY_PATH
        boost::split(paths, path, boost::is_any_of(":"));
        //trim ":" and " " from the beginning and end of the string
        for(std::string& path : paths)
        {
            boost::trim_if(path, boost::is_any_of(": "));
            boost::trim_right_if(path, boost::is_any_of("/"));
            path += plugin_files_path;
        }
    }
    return paths;
}

void PluginManager::determineAvailableXMLPluginFiles(const std::string& plugin_xml_folder, std::set<std::string>& plugin_xml_files) const
{
    boost::filesystem::path folder(plugin_xml_folder);
    if(boost::filesystem::exists(folder))
    {
        if(boost::filesystem::is_directory(folder))
        {
            for(boost::filesystem::directory_iterator it(folder); it != boost::filesystem::directory_iterator(); it++)
            {
                if(boost::filesystem::is_regular_file(*it))
                {
                    std::string extension = it->path().extension().string();
                    boost::algorithm::to_lower(extension);

                    if(extension == plugin_file_extension)
                        plugin_xml_files.insert(it->path().string());
                }
            }
        }
        else if(boost::filesystem::is_regular_file(folder))
        {
            // folder is actualy a file
            std::string extension = folder.extension().string();
            boost::algorithm::to_lower(extension);

            if(extension == plugin_file_extension)
                plugin_xml_files.insert(folder.string());
        }
    }
}

bool PluginManager::processSingleXMLPluginFile(const std::string& xml_file, std::vector< PluginInfoPtr >& class_available)
{
    TiXmlDocument document;
    document.LoadFile(xml_file);

    TiXmlElement* library = document.RootElement();
    if (library == NULL)
    {
        LOG(ERROR) << "Skipping XML Document " << xml_file << " which had no Root Element. This likely means the XML is malformed or missing.";
        return false;
    }
    if (library->ValueStr() != "library")
    {
        LOG(ERROR) << "The XML document " << xml_file << " must have \"library\" as root tag.";
        return false;
    }

    while (library != NULL)
    {
        // read library path
        const char* library_path = library->Attribute("path");
        if (library_path == NULL)
        {
            LOG(ERROR) << "Failed to find path attirbute in library element in " << xml_file;
            continue;
        }

        TiXmlElement* class_element = library->FirstChildElement("class");
        while (class_element != NULL)
        {
            PluginInfoPtr plugin_info(new PluginInfo);
            const char* base_class_name = class_element->Attribute("base_class_name");
            const char* full_class_name = class_element->Attribute("class_name");

            if(base_class_name != NULL && full_class_name != NULL)
            {
                plugin_info->full_class_name = full_class_name;
                plugin_info->base_class_name = base_class_name;
                plugin_info->library_path = library_path;
                plugin_info->class_name = removeNamespace(plugin_info->full_class_name);

                // find description
                TiXmlElement* description_element = class_element->FirstChildElement("description");
                if(description_element != NULL)
                    plugin_info->description = description_element->GetText();

                // find associations
                TiXmlElement* association_element = class_element->FirstChildElement("associations");
                if(association_element != NULL)
                {
                    TiXmlElement* associated_class_element = association_element->FirstChildElement("class");
                    while (associated_class_element != NULL)
                    {
                        const char* associated_class_name = associated_class_element->Attribute("class_name");
                        if(associated_class_name != NULL)
                            plugin_info->associated_classes.push_back(std::string(associated_class_name));
                        associated_class_element = associated_class_element->NextSiblingElement("class");
                    }
                }

                // find singleton information
                plugin_info->singleton = false;
                TiXmlElement* singleton_element = class_element->FirstChildElement("singleton");
                if(singleton_element != NULL && strcmp(singleton_element->GetText(), "true") == 0)
                {
                    plugin_info->singleton = true;
                }

                // find meta information
                TiXmlElement* meta_element = class_element->FirstChildElement("meta");
                if(meta_element != NULL)
                {
                    // parse user specific tags using a callback function
                    this->parsePluginMetaInformation(plugin_info, meta_element);
                }

                class_available.push_back(plugin_info);
            }
            else
            {
                LOG(ERROR) << "Couldn't find a valid class_name or base_class_name attribute in class element in " << xml_file;
            }
            class_element = class_element->NextSiblingElement("class");
        }
        library = library->NextSiblingElement("library");
    }

    return true;
}

void PluginManager::insertPluginInfos(const std::vector<PluginInfoPtr>& classes)
{
    for(const PluginInfoPtr &plugin_info : classes)
    {
        if(classes_available.count(plugin_info->full_class_name) == 0)
        {
            classes_available[plugin_info->full_class_name] = plugin_info;
            base_classes_available.insert(std::make_pair(plugin_info->base_class_name, plugin_info));
            classes_no_ns_available.insert(std::make_pair(plugin_info->class_name, plugin_info));
        }
        else
        {
            LOG(WARNING) << "Class " << plugin_info->full_class_name << " already available, cannot add class info twice.";
        }
    }
}

bool PluginManager::hasNamespace(const std::string& class_name) const
{
    if(hasEmbeddedType(class_name))
        return extractBaseType(class_name).find("::") != std::string::npos;
    else
        return class_name.find("::") != std::string::npos;
}

bool PluginManager::hasEmbeddedType(const std::string& class_name) const
{
    return class_name.find("<") != std::string::npos;
}

std::string PluginManager::extractEmbeddedType(const std::string& class_name) const
{
    return std::string(boost::end(boost::find_first(class_name, "<")), boost::begin(boost::find_last(class_name, ">")));
}

std::string PluginManager::extractBaseType(const std::string& class_name) const
{
    return std::string(class_name.begin(), boost::begin(boost::find_first(class_name, "<")));
}

std::string PluginManager::removeNamespace(const std::string& class_name) const
{
    // remove embedded type if necessary
    bool has_embedded_type = hasEmbeddedType(class_name);
    std::string class_name_base;
    if (has_embedded_type)
        class_name_base = extractBaseType(class_name);
    else
        class_name_base = class_name;

    // remove namespace
    std::string class_name_no_ns;
    std::vector<std::string> split_names;
    boost::split(split_names, class_name_base, boost::is_any_of("::"));
    if(!split_names.empty())
        class_name_no_ns = split_names.back();
    else
        class_name_no_ns = class_name;

    if (has_embedded_type)
        return class_name_no_ns + "<" + extractEmbeddedType(class_name) + ">";
    else
        return class_name_no_ns;
}

void PluginManager::parsePluginMetaInformation(const PluginInfoPtr& plugin_info, TiXmlElement* meta_element)
{
    // Can be implemented in inherited classes
}

bool PluginManager::getFullClassName(const std::string& class_name, std::string& full_class_name) const
{
    if(classes_available.count(class_name) >= 1)
    {
        // even if class_name doesn't have a namespace, this is all information we have
        full_class_name = class_name;
        return true;
    }
    else
    {
        size_t count = classes_no_ns_available.count(class_name);
        if(count == 1)
        {
            full_class_name = classes_no_ns_available.find(class_name)->second->full_class_name;
            return true;
        }
        else if(count == 0)
            LOG(WARNING) << "Class " << class_name << " is unknown.";
        else
            LOG(WARNING) << "Class " << class_name << " is multiple defined in different namespaces. Please use the full class name.";
    }
    return false;
}