#ifndef ADVANCED_NAVIGATION_ANPP_TEST_HELPERS_HPP
#define ADVANCED_NAVIGATION_ANPP_TEST_HELPERS_HPP

#include "gtest/gtest.h"
#include "gmock/gmock.h"
#include <imu_advanced_navigation_anpp/Protocol.hpp>
#include <imu_advanced_navigation_anpp/Driver.hpp>
#include <iodrivers_base/FixtureGTest.hpp>

inline void RAW_SET(uint8_t* begin, std::vector<uint8_t> bytes)
{
    std::copy(bytes.begin(), bytes.end(), begin);
}

// Floating-point values below have been generated by taking a  seed value
// (the first one, starting with the 123456... sequence) and then adding one
// to the first and last byte, checking the corresponding FP value
//
// Found FP-to-binary converters online
template<typename T>
struct FPValue
{
    T fp;
    std::vector<uint8_t> binary;
};

static FPValue<float> TEST_FP4_ZERO = { 0.0, { 0, 0, 0, 0 } };
static FPValue<float> TEST_FP4_ONE  = { 0.0, { 0, 0, 0x80, 0x3f } };

static FPValue<float> TEST_FP4[] = 
{
    { 1.23456792E8, { 0xA3, 0x79, 0xEB, 0x4C } },
    { 4.938272E8, { 0xA4, 0x79, 0xEB, 0x4D } },
    { 1.975308928E9, { 0xA5, 0x79, 0xEB, 0x4E } },
    { 7.901236224E9, { 0xA6, 0x79, 0xEB, 0x4F } },
    { 3.1604946944E10, { 0xA7, 0x79, 0xEB, 0x50 } },
    { 1.26419795968E11, { 0xA8, 0x79, 0xEB, 0x51 } },
    { 5.0567921664E11, { 0xA9, 0x79, 0xEB, 0x52 } },
    { 2.022716997632E12, { 0xAA, 0x79, 0xEB, 0x53 } },
    { 8.090868514816E12, { 0xAB, 0x79, 0xEB, 0x54 } },
    { 3.2363476156416E13, { 0xAC, 0x79, 0xEB, 0x55 } },
    { 1.29453913014272E14, { 0xAD, 0x79, 0xEB, 0x56 } },
    { 5.1781568561152E14, { 0xAE, 0x79, 0xEB, 0x57 } },
    { 2.071262876663808E15, { 0xAF, 0x79, 0xEB, 0x58 } },
    { 8.285052043526144E15, { 0xB0, 0x79, 0xEB, 0x59 } },
    { 1.35742301477225365504E20, { 0xB1, 0x79, 0xEB, 0x60 } },
    { 5.42969241093273550848E20, { 0xB2, 0x79, 0xEB, 0x61 } }
};

static FPValue<double> TEST_FP8[] = 
{
    { 1.2345678910111214e16, { 0xF7, 0x5E, 0xF9, 0x2E, 0x2A, 0xEE, 0x45, 0x43 } },
    { 8.09086413053048651776E20, { 0xF8, 0x5E, 0xF9, 0x2E, 0x2A, 0xEE, 0x45, 0x44 } },
    { 5.3024287165844605032726528E25, { 0xF9, 0x5E, 0xF9, 0x2E, 0x2A, 0xEE, 0x45, 0x45 } }
};

template<typename Packet>
std::vector<uint8_t> makeQuery()
{
    using namespace imu_advanced_navigation_anpp::protocol;

    std::vector<uint8_t> result(Header::SIZE + 1);
    uint8_t id = Packet::ID;
    new(&result[0]) Header(Request::ID, &id, &id + 1);
    result[Header::SIZE] = Packet::ID;
    return result;
}

template<typename Packet>
std::vector<uint8_t> makePacket(std::vector<uint8_t> const& payload)
{
    using namespace imu_advanced_navigation_anpp::protocol;

    Header header(Packet::ID, &payload[0], &payload[payload.size()]);
    uint8_t const* header_ptr = reinterpret_cast<uint8_t const*>(&header);
    std::vector<uint8_t> result;
    result.insert(result.end(), header_ptr, header_ptr + Header::SIZE);
    result.insert(result.end(), payload.begin(), payload.end());
    return result;
}

template<typename Packet>
std::vector<uint8_t> makePacket()
{
    using namespace imu_advanced_navigation_anpp::protocol;

    std::vector<uint8_t> payload(Packet::SIZE, 0);
    return makePacket<Packet>(payload);
}

inline std::vector<uint8_t> makeAcknowledge(std::vector<uint8_t> const& packet, imu_advanced_navigation_anpp::ACK_RESULTS result)
{
    return makePacket<imu_advanced_navigation_anpp::protocol::Acknowledge>(
            { packet[1], packet[3], packet[4], static_cast<uint8_t>(result) });
}

struct DriverTestBase : ::testing::Test, iodrivers_base::Fixture<imu_advanced_navigation_anpp::Driver>
{
    void openTestURI()
    { IODRIVERS_BASE_MOCK();
        using namespace imu_advanced_navigation_anpp::protocol;

        std::vector<uint8_t> clearPacket =
            makePacket<PacketPeriods>({ 0, 1, UnixTime::ID, 0, 0, 0, 0 });
        EXPECT_REPLY(
                clearPacket,
                makeAcknowledge(clearPacket, imu_advanced_navigation_anpp::ACK_SUCCESS));
        driver.openURI("test://");
    }

    void EXPECT_PACKET_PERIOD(uint8_t packet_id, uint8_t period)
    {
        using namespace imu_advanced_navigation_anpp::protocol;
        std::vector<uint8_t> packet =
            makePacket<PacketPeriods>({ 0, 0, packet_id, period, 0, 0, 0 });
        EXPECT_REPLY(
                packet,
                makeAcknowledge(packet, imu_advanced_navigation_anpp::ACK_SUCCESS));
    }

};

#endif
