require 'models/blueprints/control'
require 'models/blueprints/pose'
require 'models/blueprints/planning'

# Integration of the trajectory follower component
#
# It can be used in a ControlLoop composition as a Motion2DCommand provider.
# In this case, the corresponding specialization is used, which requires a
# Base::PoseSrv provider.
#
# It can also be used with a static trajectory. One can set the following two
# configuration parameters:
#
#   Conf.trajectory_file = path_to_trajectory_file
#
# where the trajectory file contains a trajectory (i.e.
# std::vector<wrappers::Waypoint> marshalled by typelib. It can be generated by
# getting a data sample and doing
#
#   File.open('file_name', 'w') do |io|
#     io.write(trajectory_sample.to_byte_array)
#   end
#
# Moreover, if the Conf.reverse_trajectory flag is set, the trajectory follower
# will follow the static trajectory in the reverse direction
class TrajectoryFollower::Task
    provides Base::Motion2DControllerSrv, :as => 'controller'

    argument :trajectory, :default => nil

    # Add some more information for the transformer integration
    transformer do
        associate_frame_to_ports "world", "trajectory"
        transform_input "robot_pose", "body" => "world"
    end

    script do
        trajectory_w = trajectory_port.writer
        wait_until_ready trajectory_w
        execute do
            if trajectory
                trajectory_w.write trajectory
            end
        end
    end
end

Base::ControlLoop.specialize \
    Base::ControlLoop.controller_child => TrajectoryFollower::Task,
    Base::ControlLoop.controlled_system_child => Base::Motion2DControlledSystemSrv do

    add Base::PoseSrv, :as => 'pose'
    pose_child.connect_to controller_child
    export controller_child.trajectory_port

    provides Rock::Planning::TrajectoryExecutionSrv, :as => 'trajectory_execution'
end

